#!/usr/bin/env python3
"""
run_kernel.py

Builds, for each context index i, a tridiagonal reproduction-kernel matrix M^(i)
based solely on D_i, computes its dominant eigenvalue rho_i, and writes out:
  - kernel_eigs.csv
  - rho_vs_D.png
  - M_heatmap.png
"""

import numpy as np
import pandas as pd
import matplotlib.pyplot as plt
from pathlib import Path
from scipy import linalg as la

# 1. Load inputs (ensure UTF-8 read)
Path("vol3_kernel_spec.txt").read_text(encoding="utf-8")  # spec currently unused
df = pd.read_csv("D_values.csv")  # columns: "n","D"
N = len(df)

# 2. Pivot function
def g(D):
    return D / 3.0

# 3. Container for spectral radii
rhos = np.zeros(N)

# 4. Build M^(i) for each D_i and compute rho_i
for idx, Di in enumerate(df["D"]):
    gi = g(Di)
    # Create empty tridiagonal matrix
    M = np.zeros((N, N))
    # Fill diagonal and off-diagonals based on Di, gi
    for j in range(N):
        M[j, j] = Di - 2 * gi
        if j > 0:
            M[j, j-1] = gi
        if j < N - 1:
            M[j, j+1] = gi

    # Compute spectral radius (largest absolute eigenvalue)
    eigs = la.eigvals(M)
    rhos[idx] = np.max(np.abs(eigs))

# 5. Save results table
out = df.copy()
out["rho"] = rhos
out.to_csv("kernel_eigs.csv", index=False)

# 6. Plot rho vs D
plt.figure(figsize=(6, 4))
plt.scatter(out["D"], out["rho"], s=60, label="ρ(M^(i))")
mn, mx = out["D"].min(), out["D"].max()
plt.plot([mn, mx], [mn, mx], ls="--", label="y = x")
plt.xlabel("D(n)")
plt.ylabel("ρ(M)")
plt.title("Spectral Radius vs Fractal Dimension")
plt.legend()
plt.tight_layout()
plt.savefig("rho_vs_D.png", dpi=150)

# 7. Heatmap of M for the median index
mid = N // 2
Di_mid = df["D"].iat[mid]
gi_mid = g(Di_mid)
M_mid = np.zeros((N, N))
for j in range(N):
    M_mid[j, j] = Di_mid - 2 * gi_mid
    if j > 0:
        M_mid[j, j-1] = gi_mid
    if j < N - 1:
        M_mid[j, j+1] = gi_mid

plt.figure(figsize=(6, 5))
plt.imshow(M_mid, cmap="viridis", origin="lower")
plt.colorbar(label="M value")
plt.title(f"Heatmap of M for n={df['n'].iat[mid]}, D={Di_mid:.2f}")
plt.tight_layout()
plt.savefig("M_heatmap.png", dpi=150)

print("✅ Done: kernel_eigs.csv, rho_vs_D.png, M_heatmap.png")
